<?php
/**
 * Metrics
 * File ini berisi fungsi-fungsi untuk menghitung metrik evaluasi model
 */

/**
 * Menghitung Mean Absolute Error (MAE)
 * 
 * @param array $actual Nilai aktual
 * @param array $predicted Nilai prediksi
 * @return float MAE
 */
function calculateMAE($actual, $predicted) {
    if (count($actual) != count($predicted) || count($actual) == 0) {
        return 0;
    }
    
    $sum = 0;
    
    for ($i = 0; $i < count($actual); $i++) {
        $sum += abs($actual[$i] - $predicted[$i]);
    }
    
    return $sum / count($actual);
}

/**
 * Menghitung Root Mean Square Error (RMSE)
 * 
 * @param array $actual Nilai aktual
 * @param array $predicted Nilai prediksi
 * @return float RMSE
 */
function calculateRMSE($actual, $predicted) {
    if (count($actual) != count($predicted) || count($actual) == 0) {
        return 0;
    }
    
    $sum = 0;
    
    for ($i = 0; $i < count($actual); $i++) {
        $sum += pow($actual[$i] - $predicted[$i], 2);
    }
    
    return sqrt($sum / count($actual));
}

/**
 * Menghitung Mean Absolute Percentage Error (MAPE)
 * 
 * @param array $actual Nilai aktual
 * @param array $predicted Nilai prediksi
 * @return float MAPE
 */
function calculateMAPE($actual, $predicted) {
    if (count($actual) != count($predicted) || count($actual) == 0) {
        return 0;
    }
    
    $sum = 0;
    $count = 0;
    
    for ($i = 0; $i < count($actual); $i++) {
        if ($actual[$i] != 0) {
            $sum += abs(($actual[$i] - $predicted[$i]) / $actual[$i]);
            $count++;
        }
    }
    
    return ($count > 0) ? ($sum / $count * 100) : 0;
}

/**
 * Menghitung R-squared (Coefficient of Determination)
 * 
 * @param array $actual Nilai aktual
 * @param array $predicted Nilai prediksi
 * @return float R-squared
 */
function calculateRSquared($actual, $predicted) {
    if (count($actual) != count($predicted) || count($actual) == 0) {
        return 0;
    }
    
    // Hitung rata-rata nilai aktual
    $mean = array_sum($actual) / count($actual);
    
    // Hitung total sum of squares (SST)
    $sst = 0;
    foreach ($actual as $value) {
        $sst += pow($value - $mean, 2);
    }
    
    // Hitung residual sum of squares (SSR)
    $ssr = 0;
    for ($i = 0; $i < count($actual); $i++) {
        $ssr += pow($actual[$i] - $predicted[$i], 2);
    }
    
    // R-squared = 1 - (SSR / SST)
    return ($sst != 0) ? (1 - ($ssr / $sst)) : 0;
}

/**
 * Mengekstrak nilai actual dan predicted dari array prediksi
 * 
 * @param array $predictions Array prediksi
 * @return array Array actual dan predicted
 */
function extractActualAndPredicted($predictions) {
    $actual = [];
    $predicted = [];
    
    foreach ($predictions as $prediction) {
        $actual[] = $prediction['actual_price'];
        $predicted[] = $prediction['predicted_price'];
    }
    
    return [
        'actual' => $actual,
        'predicted' => $predicted
    ];
}

/**
 * Menghitung semua metrik evaluasi
 * 
 * @param array $predictions Array prediksi
 * @return array Metrik evaluasi
 */
function calculateAllMetrics($predictions) {
    $data = extractActualAndPredicted($predictions);
    $actual = $data['actual'];
    $predicted = $data['predicted'];
    
    return [
        'mae' => calculateMAE($actual, $predicted),
        'rmse' => calculateRMSE($actual, $predicted),
        'mape' => calculateMAPE($actual, $predicted),
        'r_squared' => calculateRSquared($actual, $predicted)
    ];
}