<?php
/**
 * CSV Handler
 * File ini berisi fungsi-fungsi untuk menangani file CSV
 */

/**
 * Membaca data dari file CSV
 * 
 * @param string $filePath Path file CSV
 * @return array|false Data dari CSV jika berhasil, false jika gagal
 */
function readCSV($filePath) {
    if (!file_exists($filePath)) {
        return false;
    }
    
    $data = [];
    
    // Buka file CSV
    if (($handle = fopen($filePath, "r")) !== FALSE) {
        // Baca header
        $header = fgetcsv($handle, 0, ",");
        
        // Standardisasi header
        $header = array_map('trim', $header);
        
        // Baca data
        while (($row = fgetcsv($handle, 0, ",")) !== FALSE) {
            if (count($row) == count($header)) {
                $data[] = array_combine($header, $row);
            }
        }
        
        fclose($handle);
    } else {
        return false;
    }
    
    return $data;
}

/**
 * Memvalidasi data CSV saham
 * 
 * @param array $data Data dari CSV
 * @return array|false Data yang sudah divalidasi jika valid, false jika tidak
 */
function validateStockData($data) {
    if (empty($data)) {
        return false;
    }
    
    $validatedData = [];
    
    // Cek format header yang mungkin (lama atau baru)
    $firstRow = $data[0];
    $headerFormat = determineHeaderFormat($firstRow);
    
    if ($headerFormat === false) {
        return false;
    }
    
    // Validasi setiap baris
    foreach ($data as $row) {
        // Sesuaikan dengan format header yang terdeteksi
        if ($headerFormat === 'new') {
            // Format baru: Date, Open, High, Low, Close, Volume, Adj Close
            $date = isset($row['Date']) ? $row['Date'] : '';
            $open = isset($row['Open']) ? $row['Open'] : 0;
            $high = isset($row['High']) ? $row['High'] : 0;
            $low = isset($row['Low']) ? $row['Low'] : 0;
            $close = isset($row['Close']) ? $row['Close'] : 0;
            $volume = isset($row['Volume']) ? $row['Volume'] : 0;
            $adjClose = isset($row['Adj Close']) ? $row['Adj Close'] : (isset($row['Adjusted Close']) ? $row['Adjusted Close'] : $close);
        } else {
            // Format lama: Date, Open, High, Low, Close, Adj Close, Volume
            $date = isset($row['Date']) ? $row['Date'] : '';
            $open = isset($row['Open']) ? $row['Open'] : 0;
            $high = isset($row['High']) ? $row['High'] : 0;
            $low = isset($row['Low']) ? $row['Low'] : 0;
            $close = isset($row['Close']) ? $row['Close'] : 0;
            $adjClose = isset($row['Adj Close']) ? $row['Adj Close'] : $close;
            $volume = isset($row['Volume']) ? $row['Volume'] : 0;
        }
        
        // Validasi tanggal
        $date = standardizeDate($date);
        if (empty($date)) {
            continue;
        }
        
        // Validasi nilai numerik
        if (!is_numeric($open) || !is_numeric($high) || !is_numeric($low) || 
            !is_numeric($close) || !is_numeric($adjClose) || !is_numeric($volume)) {
            continue;
        }
        
        $validatedData[] = [
            'Date' => $date,
            'Open' => (float) $open,
            'High' => (float) $high,
            'Low' => (float) $low,
            'Close' => (float) $close,
            'Adj Close' => (float) $adjClose,
            'Volume' => (int) $volume
        ];
    }
    
    return !empty($validatedData) ? $validatedData : false;
}

/**
 * Menentukan format header CSV (lama atau baru)
 * 
 * @param array $row Baris pertama data CSV
 * @return string|false 'new', 'old', atau false jika format tidak valid
 */
function determineHeaderFormat($row) {
    // Cek keberadaan kolom wajib
    $requiredColumns = ['Date', 'Open', 'High', 'Low', 'Close', 'Volume'];
    foreach ($requiredColumns as $column) {
        if (!array_key_exists($column, $row)) {
            return false;
        }
    }
    
    // Cek posisi Volume untuk menentukan format
    $keysArray = array_keys($row);
    $adjCloseIndex = array_search('Adj Close', $keysArray);
    $volumeIndex = array_search('Volume', $keysArray);
    
    if ($adjCloseIndex === false) {
        // Cek jika menggunakan nama kolom "Adjusted Close" sebagai alternatif
        $adjCloseIndex = array_search('Adjusted Close', $keysArray);
    }
    
    if ($adjCloseIndex !== false && $volumeIndex !== false) {
        // Format baru: Volume sebelum Adj Close
        if ($volumeIndex < $adjCloseIndex) {
            return 'new';
        } 
        // Format lama: Adj Close sebelum Volume
        else {
            return 'old';
        }
    }
    
    // Default ke format baru jika tidak bisa menentukan dengan jelas
    return 'new';
}

/**
 * Standardisasi format tanggal
 * 
 * @param string $date Tanggal dalam berbagai format
 * @return string Format tanggal Y-m-d
 */
function standardizeDate($date) {
    // Coba beberapa format tanggal yang mungkin
    $formats = ['Y-m-d', 'd/m/Y', 'm/d/Y', 'd-m-Y', 'm-d-Y'];
    
    foreach ($formats as $format) {
        $d = DateTime::createFromFormat($format, $date);
        if ($d && $d->format($format) == $date) {
            return $d->format('Y-m-d');
        }
    }
    
    // Jika tidak cocok dengan format yang sudah ditentukan, coba parse tanggal
    $timestamp = strtotime($date);
    if ($timestamp !== false) {
        return date('Y-m-d', $timestamp);
    }
    
    return '';
}

/**
 * Menulis data ke file CSV
 * 
 * @param string $filePath Path file CSV
 * @param array $data Data yang akan ditulis
 * @param array $header Header file CSV
 * @return bool True jika berhasil, false jika gagal
 */
// function writeCSV($filePath, $data, $header) {
//     // Buka file untuk ditulis
//     $handle = fopen($filePath, 'w');
    
//     if ($handle === false) {
//         return false;
//     }
    
//     // Tulis header
//     fputcsv($handle, $header);
    
//     // Tulis data
//     foreach ($data as $row) {
//         fputcsv($handle, $row);
//     }
    
//     fclose($handle);
    
//     return true;
// }

/**
 * Menyiapkan data untuk export
 * 
 * @param array $predictions Data prediksi
 * @return array Data yang siap di-export
 */
function prepareExportData($predictions) {
    $exportData = [];
    
    foreach ($predictions as $prediction) {
        $exportData[] = [
            $prediction['date'],
            $prediction['actual_price'],
            $prediction['predicted_price'],
            $prediction['error']
        ];
    }
    
    return $exportData;
}