<?php
/**
 * Model Evaluation
 * File ini berisi fungsi-fungsi terkait evaluasi model
 */

require_once __DIR__ . '/../config/database.php';

/**
 * Menyimpan evaluasi model
 * 
 * @param int $modelId ID model
 * @param float $mae Mean Absolute Error
 * @param float $rmse Root Mean Squared Error
 * @return bool True jika berhasil, false jika gagal
 */
function saveEvaluation($modelId, $mae, $rmse) {
    global $conn;
    
    // Hapus evaluasi yang ada dengan model ID yang sama
    $sql = "DELETE FROM model_evaluation WHERE model_id = $modelId";
    executeQuery($conn, $sql);
    
    // Simpan evaluasi baru
    $sql = "INSERT INTO model_evaluation (model_id, mae, rmse) VALUES ($modelId, $mae, $rmse)";
    executeQuery($conn, $sql);
    
    return mysqli_affected_rows($conn) > 0;
}

/**
 * Mendapatkan evaluasi model berdasarkan ID model
 * 
 * @param int $modelId ID model
 * @return array|false Data evaluasi jika ditemukan, false jika tidak
 */
function getEvaluationByModelId($modelId) {
    global $conn;
    
    $sql = "SELECT * FROM model_evaluation WHERE model_id = $modelId";
    $result = executeQuery($conn, $sql);
    
    if (mysqli_num_rows($result) > 0) {
        return fetchRow($result);
    }
    
    return false;
}

/**
 * Mendapatkan semua evaluasi model
 * 
 * @param int $limit Batas jumlah data yang diambil
 * @return array Data evaluasi model
 */
function getAllEvaluations($limit = 10) {
    global $conn;
    
    $sql = "SELECT me.*, ms.train_test_ratio, ms.n_estimators, ms.created_at 
            FROM model_evaluation me 
            JOIN model_settings ms ON me.model_id = ms.id 
            ORDER BY me.created_at DESC 
            LIMIT $limit";
    
    $result = executeQuery($conn, $sql);
    
    return fetchData($result);
}

/**
 * Mendapatkan evaluasi model terbaik
 * 
 * @param string $metric Metrik yang digunakan ('mae' atau 'rmse')
 * @return array|false Data evaluasi model terbaik jika ditemukan, false jika tidak
 */
function getBestEvaluation($metric = 'rmse') {
    global $conn;
    
    $orderBy = $metric == 'mae' ? 'me.mae ASC' : 'me.rmse ASC';
    
    $sql = "SELECT me.*, ms.train_test_ratio, ms.n_estimators, ms.created_at 
            FROM model_evaluation me 
            JOIN model_settings ms ON me.model_id = ms.id 
            ORDER BY $orderBy 
            LIMIT 1";
    
    $result = executeQuery($conn, $sql);
    
    if (mysqli_num_rows($result) > 0) {
        return fetchRow($result);
    }
    
    return false;
}

/**
 * Menghapus evaluasi model berdasarkan ID model
 * 
 * @param int $modelId ID model
 * @return bool True jika berhasil, false jika gagal
 */
function deleteEvaluation($modelId) {
    global $conn;
    
    $sql = "DELETE FROM model_evaluation WHERE model_id = $modelId";
    executeQuery($conn, $sql);
    
    return mysqli_affected_rows($conn) > 0;
}

/**
 * Bandingkan evaluasi antara dua model
 * 
 * @param int $modelId1 ID model pertama
 * @param int $modelId2 ID model kedua
 * @return array Hasil perbandingan
 */
function compareEvaluations($modelId1, $modelId2) {
    $eval1 = getEvaluationByModelId($modelId1);
    $eval2 = getEvaluationByModelId($modelId2);
    
    if (!$eval1 || !$eval2) {
        return false;
    }
    
    $maeDiff = $eval1['mae'] - $eval2['mae'];
    $rmseDiff = $eval1['rmse'] - $eval2['rmse'];
    
    $maeImprovement = ($eval2['mae'] != 0) ? ($maeDiff / $eval2['mae'] * 100) : 0;
    $rmseImprovement = ($eval2['rmse'] != 0) ? ($rmseDiff / $eval2['rmse'] * 100) : 0;
    
    return [
        'model1' => [
            'id' => $modelId1,
            'mae' => $eval1['mae'],
            'rmse' => $eval1['rmse']
        ],
        'model2' => [
            'id' => $modelId2,
            'mae' => $eval2['mae'],
            'rmse' => $eval2['rmse']
        ],
        'diff' => [
            'mae' => $maeDiff,
            'rmse' => $rmseDiff
        ],
        'improvement' => [
            'mae' => $maeImprovement,
            'rmse' => $rmseImprovement
        ]
    ];
}

/**
 * Mendapatkan statistik evaluasi model (rata-rata, min, max)
 * 
 * @return array Statistik evaluasi
 */
function getEvaluationStats() {
    global $conn;
    
    $sql = "SELECT 
                AVG(mae) as avg_mae, 
                MIN(mae) as min_mae, 
                MAX(mae) as max_mae, 
                AVG(rmse) as avg_rmse, 
                MIN(rmse) as min_rmse, 
                MAX(rmse) as max_rmse,
                COUNT(*) as total_models
            FROM model_evaluation";
    
    $result = executeQuery($conn, $sql);
    
    if (mysqli_num_rows($result) > 0) {
        return fetchRow($result);
    }
    
    return [
        'avg_mae' => 0,
        'min_mae' => 0,
        'max_mae' => 0,
        'avg_rmse' => 0,
        'min_rmse' => 0,
        'max_rmse' => 0,
        'total_models' => 0
    ];
}