<?php
/**
 * Prediction Controller
 * File ini menangani proses prediksi saham
 */
ob_start();
// Aktifkan pelaporan semua jenis error
error_reporting(E_ALL);

// Tampilkan error ke browser
ini_set('display_errors', 1);

// Untuk environment development
ini_set('display_startup_errors', 1);


require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/session.php';
require_once __DIR__ . '/../model/stock_model.php';
require_once __DIR__ . '/../model/prediction_model.php';
require_once __DIR__ . '/../model/settings_model.php';
require_once __DIR__ . '/../utils/random_forest.php';
require_once __DIR__ . '/../utils/visualize.php';
require_once __DIR__ . '/../utils/csv_handler.php';
require_once __DIR__ . '/../utils/metrics.php';

// Definisikan BASE_URL jika belum didefinisikan
if (!defined('BASE_URL')) {
    define('BASE_URL', '/bri_stock_prediction');
}

/**
 * Menangani proses training model dan prediksi
 */
function handleTraining() {
    // Cek apakah user sudah login
    if (!isLoggedIn()) {
        header('Location: ' . BASE_URL . '/view/auth/login.php');
        exit;
    }
    
    // Cek apakah form sudah disubmit
    if ($_SERVER['REQUEST_METHOD'] == 'POST') {
        // Dapatkan parameter model
        $trainTestRatio = isset($_POST['train_test_ratio']) ? (float) $_POST['train_test_ratio'] : 0.8;
        $nEstimators = isset($_POST['n_estimators']) ? (int) $_POST['n_estimators'] : 100;
        $maxDepth = isset($_POST['max_depth']) && !empty($_POST['max_depth']) ? (int) $_POST['max_depth'] : NULL;
        $minSamplesSplit = isset($_POST['min_samples_split']) ? (int) $_POST['min_samples_split'] : 2;
        $minSamplesLeaf = isset($_POST['min_samples_leaf']) ? (int) $_POST['min_samples_leaf'] : 1;
        
        // Validasi parameter
        if ($trainTestRatio <= 0 || $trainTestRatio >= 1) {
            setFlashMessage('error', 'Rasio training-testing harus antara 0 dan 1');
            return;
        }
        
        if ($nEstimators <= 0) {
            setFlashMessage('error', 'Jumlah estimator harus lebih dari 0');
            return;
        }
        
        // Simpan pengaturan model
        $modelId = saveModelSettings(
            $trainTestRatio,
            $nEstimators,
            $maxDepth,
            $minSamplesSplit,
            $minSamplesLeaf
        );
        
        // Dapatkan data saham
        $stockData = getClosePriceData();
        
        if (empty($stockData)) {
            setFlashMessage('error', 'Data saham kosong, upload data terlebih dahulu');
            return;
        }
        
        // Dapatkan pengaturan model
        $modelSettings = getModelSettingsById($modelId);
        
        // Jalankan Random Forest
        $results = runRandomForest($stockData, $modelSettings);
        
        if ($results) {
            // Simpan prediksi
            savePredictions($results['predictions'], $modelId);
            
            // Simpan evaluasi model
            saveModelEvaluation(
                $modelId,
                $results['metrics']['mae'],
                $results['metrics']['rmse']
            );
            
            setFlashMessage('success', 'Training model berhasil');
            
            // Redirect ke halaman hasil
            header('Location: ' . BASE_URL . '/view/prediction/result.php?model_id=' . $modelId);
            exit;
        } else {
            setFlashMessage('error', 'Gagal melakukan training model');
        }
    }
}

/**
 * Mendapatkan hasil prediksi
 * 
 * @param int $modelId ID model
 * @return array Hasil prediksi
 */
function getPredictionResults($modelId = null) {
    // Jika model ID tidak disediakan, gunakan model terbaru
    if ($modelId === null) {
        $modelSettings = getLatestModelSettings();
        $modelId = $modelSettings['id'];
    } else {
        $modelSettings = getModelSettingsById($modelId);
    }
    
    // Dapatkan prediksi
    $predictions = getPredictionsByModelId($modelId);
    
    // Dapatkan evaluasi model
    $evaluation = getModelEvaluationByModelId($modelId);
    
    // Format data untuk chart
    $predictionChartData = formatPredictionChartData($predictions);
    $errorChartData = formatErrorChartData($predictions);
    
    return [
        'model_settings' => $modelSettings,
        'predictions' => $predictions,
        'evaluation' => $evaluation,
        'prediction_chart_data' => json_encode($predictionChartData),
        'error_chart_data' => json_encode($errorChartData)
    ];
}

/**
 * Menulis data ke file CSV
 * 
 * @param string $filePath Path file CSV
 * @param array $data Data yang akan ditulis
 * @param array $header Header file CSV
 * @return bool True jika berhasil, false jika gagal
 */
function writeCSV($filePath, $data, $header) {
    // Buka file untuk ditulis
    $handle = fopen($filePath, 'w');
    
    if ($handle === false) {
        return false;
    }
    
    // Tambahkan BOM untuk Excel agar mengenali UTF-8
    fprintf($handle, chr(0xEF).chr(0xBB).chr(0xBF));
    
    // Tulis header
    fputcsv($handle, $header, ';'); // Gunakan ; sebagai delimiter untuk Excel
    
    // Tulis data dengan format khusus untuk angka
    foreach ($data as $row) {
        // Format angka dengan mengubah . menjadi , untuk Excel
        $formattedRow = [];
        foreach ($row as $key => $value) {
            if (is_numeric($value)) {
                // Format angka dengan awalan tanda sama dengan dan tanda kutip
                // Ini memaksa Excel mengenali sebagai teks, lalu bisa dikonversi ke angka
                $formattedRow[$key] = '="' . str_replace('.', ',', $value) . '"';
            } else {
                $formattedRow[$key] = $value;
            }
        }
        
        fputcsv($handle, $formattedRow, ';');
    }
    
    fclose($handle);
    
    return true;
}

/**
 * Menangani proses export hasil prediksi
 * 
 * @param int $modelId ID model
 */
function handleExport($modelId = null) {
    // Cek apakah user sudah login
    if (!isLoggedIn()) {
        header('Location: ' . BASE_URL . '/view/auth/login.php');
        exit;
    }
    
    // Jika model ID tidak disediakan, gunakan model terbaru
    if ($modelId === null) {
        $modelSettings = getLatestModelSettings();
        $modelId = $modelSettings['id'];
    }
    
    // Dapatkan prediksi
    $predictions = getPredictionsByModelId($modelId);
    
    if (empty($predictions)) {
        setFlashMessage('error', 'Data prediksi kosong');
        return;
    }
    
    // Buat direktori export jika belum ada
    $exportDir = __DIR__ . '/../exports/';
    if (!is_dir($exportDir)) {
        mkdir($exportDir, 0755, true);
    }
    
    // Tentukan nama file
    $filename = 'prediction_results_' . date('YmdHis') . '.csv';
    $targetFile = $exportDir . $filename;
    
    // Header file CSV
    $header = ['Tanggal', 'Harga Aktual', 'Harga Prediksi', 'Error', 'Error (%)'];
    
    // Siapkan data untuk export
    $exportData = [];
    foreach ($predictions as $prediction) {
        // Hitung error persen
        $errorPercent = ($prediction['actual_price'] > 0) 
            ? ($prediction['error'] / $prediction['actual_price'] * 100) 
            : 0;
            
        $exportData[] = [
            $prediction['date'], // Tanggal
            $prediction['actual_price'], // Harga Aktual
            $prediction['predicted_price'], // Harga Prediksi
            $prediction['error'], // Error
            number_format($errorPercent, 2) // Error (%)
        ];
    }
    
    // Tulis ke file CSV dengan format yang cocok untuk Excel
    if (writeCSV($targetFile, $exportData, $header)) {
        // Set header untuk download
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        header('Pragma: no-cache');
        readfile($targetFile);
        exit;
    } else {
        setFlashMessage('error', 'Gagal membuat file export');
    }
}

/**
 * Mendapatkan prediksi untuk hari berikutnya
 * 
 * @param int $modelId ID model yang digunakan
 * @return array Prediksi untuk hari berikutnya
 */
function getFuturePrediction($modelId = null) {
    // Jika model ID tidak disediakan, gunakan model terbaru
    if ($modelId === null) {
        $modelSettings = getLatestModelSettings();
        $modelId = $modelSettings['id'];
    } else {
        $modelSettings = getModelSettingsById($modelId);
    }
    
    // Dapatkan data saham terbaru untuk input
    $latestData = getLatestData(10); // Ambil 10 data terakhir
    
    // Siapkan feature untuk prediksi
    $lookback = 5; // Sesuaikan dengan implementasi di random_forest.php
    
    // Pastikan ada cukup data
    if (count($latestData) < $lookback) {
        return [
            'success' => false,
            'message' => 'Data tidak cukup untuk prediksi'
        ];
    }
    
    // Ambil 5 data terbaru untuk input prediksi
    $features = [];
    for ($i = 0; $i < $lookback; $i++) {
        $features[] = (float) $latestData[$i]['close'];
    }
    
    // Balik array agar urutan dari terlama ke terbaru
    $features = array_reverse($features);
    
    // Tanggal besok
    $tomorrow = date('Y-m-d', strtotime($latestData[0]['date'] . ' +1 day'));
    
    // Gunakan fungsi prediksi dari random_forest.php
    $result = predictSingleDay($features, $modelSettings);
    
    if ($result) {
        return [
            'success' => true,
            'date' => $tomorrow,
            'prediction' => $result,
            'features' => $features,
            'latest_data' => $latestData[0]
        ];
    } else {
        return [
            'success' => false,
            'message' => 'Gagal melakukan prediksi'
        ];
    }
}

/**
 * Memprediksi harga untuk beberapa hari ke depan
 * 
 * @param int $days Jumlah hari yang akan diprediksi
 * @param int $modelId ID model yang digunakan
 * @return array Prediksi untuk beberapa hari ke depan
 */
function getMultiDayPrediction($days = 7, $modelId = null) {
    // Jika model ID tidak disediakan, gunakan model terbaru
    if ($modelId === null) {
        $modelSettings = getLatestModelSettings();
        $modelId = $modelSettings['id'];
    } else {
        $modelSettings = getModelSettingsById($modelId);
    }
    
    // Dapatkan data saham terbaru
    $latestData = getLatestData(10);
    
    // Siapkan feature untuk prediksi
    $lookback = 5;
    
    // Pastikan ada cukup data
    if (count($latestData) < $lookback) {
        return [
            'success' => false,
            'message' => 'Data tidak cukup untuk prediksi'
        ];
    }
    
    // Ambil 5 data terbaru untuk input prediksi awal
    $features = [];
    for ($i = 0; $i < $lookback; $i++) {
        $features[] = (float) $latestData[$i]['close'];
    }
    
    // Balik array agar urutan dari terlama ke terbaru
    $features = array_reverse($features);
    
    // Inisialisasi array untuk menyimpan tanggal dan prediksi
    $predictions = [];
    $currentDate = $latestData[0]['date'];
    
    // Lakukan prediksi untuk setiap hari
    for ($i = 1; $i <= $days; $i++) {
        // Tanggal prediksi
        $predictionDate = date('Y-m-d', strtotime($currentDate . " +$i day"));
        
        // Gunakan fungsi prediksi dari random_forest.php
        $predictedValue = predictSingleDay($features, $modelSettings);
        
        if ($predictedValue) {
            // Tambahkan prediksi ke array
            $predictions[] = [
                'date' => $predictionDate,
                'prediction' => $predictedValue
            ];
            
            // Update features untuk prediksi berikutnya (bergeser maju satu hari)
            array_shift($features); // Buang hari terlama
            array_push($features, $predictedValue); // Tambahkan hasil prediksi
        } else {
            // Jika prediksi gagal, hentikan loop
            break;
        }
    }
    
    return [
        'success' => count($predictions) > 0,
        'predictions' => $predictions,
        'model_id' => $modelId
    ];
    ob_end_flush();
}