<?php
/**
 * Dashboard Controller
 * File ini menangani tampilan dashboard
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/session.php';
require_once __DIR__ . '/../model/stock_model.php';
require_once __DIR__ . '/../model/prediction_model.php';
require_once __DIR__ . '/../utils/visualize.php';

/**
 * Mendapatkan data untuk dashboard
 * 
 * @return array Data untuk dashboard
 */
function getDashboardData() {
    // Cek apakah user sudah login
    if (!isLoggedIn()) {
        header('Location: /view/auth/login.php');
        exit;
    }
    
    // Dapatkan jumlah data saham
    $stockDataCount = getStockDataCount();
    
    // Dapatkan data terbaru untuk visualisasi
    $latestData = getLatestData(30); // 30 hari terakhir
    
    // Dapatkan data prediksi terbaru
    $latestPredictions = getLatestPredictions();
    
    // Dapatkan evaluasi model terbaru
    $latestEvaluation = getLatestModelEvaluation();
    
    // Format data untuk chart
    $stockChartData = formatStockChartData($latestData);
    $predictionChartData = !empty($latestPredictions) ? formatPredictionChartData($latestPredictions) : null;
    
    // Kombinasikan semua data
    return [
        'stock_count' => $stockDataCount,
        'latest_data' => $latestData,
        'latest_predictions' => $latestPredictions,
        'latest_evaluation' => $latestEvaluation,
        'stock_chart_data' => json_encode($stockChartData),
        'prediction_chart_data' => $predictionChartData ? json_encode($predictionChartData) : null
    ];
}

/**
 * Mendapatkan ringkasan statistik
 * 
 * @return array Ringkasan statistik
 */
function getStatisticsSummary() {
    // Dapatkan data terbaru
    $latestData = getLatestData(30); // 30 hari terakhir
    
    if (empty($latestData)) {
        return [
            'min_price' => 0,
            'max_price' => 0,
            'avg_price' => 0,
            'last_price' => 0,
            'price_change' => 0,
            'price_change_percent' => 0
        ];
    }
    
    // Hitung statistik
    $closePrices = array_column($latestData, 'close');
    $minPrice = min($closePrices);
    $maxPrice = max($closePrices);
    $avgPrice = array_sum($closePrices) / count($closePrices);
    
    // Harga terbaru
    $lastPrice = end($closePrices);
    
    // Perubahan harga
    $firstPrice = reset($closePrices);
    $priceChange = $lastPrice - $firstPrice;
    $priceChangePercent = ($firstPrice != 0) ? ($priceChange / $firstPrice * 100) : 0;
    
    return [
        'min_price' => $minPrice,
        'max_price' => $maxPrice,
        'avg_price' => $avgPrice,
        'last_price' => $lastPrice,
        'price_change' => $priceChange,
        'price_change_percent' => $priceChangePercent
    ];
}