<?php
/**
 * Auth Controller
 * File ini menangani proses autentikasi
 */

require_once __DIR__ . '/../config/database.php';
require_once __DIR__ . '/../config/session.php';
require_once __DIR__ . '/../model/user_model.php';

// Definisikan base URL untuk redirect
define('BASE_URL', '/bri_stock_prediction');

/**
 * Menangani proses login
 */
function handleLogin() {
    // Cek apakah form sudah disubmit
    if ($_SERVER['REQUEST_METHOD'] == 'POST') {
        $username = isset($_POST['username']) ? trim($_POST['username']) : '';
        $password = isset($_POST['password']) ? $_POST['password'] : '';
        
        // Validasi input
        if (empty($username) || empty($password)) {
            setFlashMessage('error', 'Username dan password tidak boleh kosong');
            return;
        }
        
        // Debug: Simpan informasi ke log
        error_log("Login attempt: Username={$username}");
        
        // Verifikasi user
        $user = verifyUser($username, $password);
        
        if ($user) {
            // Set session
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            
            // Debug: Simpan informasi ke log
            error_log("Login successful for user: {$user['username']} with ID: {$user['id']}");
            
            setFlashMessage('success', 'Login berhasil');
            
            // Redirect ke dashboard - gunakan BASE_URL
            header('Location: ' . BASE_URL . '/view/dashboard/index.php');
            exit;
        } else {
            // Debug: Simpan informasi ke log
            error_log("Login failed for username: {$username}");
            
            setFlashMessage('error', 'Username atau password salah');
        }
    }
}

/**
 * Menangani proses logout
 */
function handleLogout() {
    // Debug: Log logout
    if (isset($_SESSION['username'])) {
        error_log("User logged out: {$_SESSION['username']}");
    }
    
    logout();
    
    // Redirect ke login dengan BASE_URL
    header('Location: ' . BASE_URL . '/view/auth/login.php');
    exit;
}

/**
 * Menangani perubahan password
 */
function handleChangePassword() {
    // Cek apakah user sudah login
    if (!isLoggedIn()) {
        header('Location: ' . BASE_URL . '/view/auth/login.php');
        exit;
    }
    
    // Cek apakah form sudah disubmit
    if ($_SERVER['REQUEST_METHOD'] == 'POST') {
        $currentPassword = isset($_POST['current_password']) ? $_POST['current_password'] : '';
        $newPassword = isset($_POST['new_password']) ? $_POST['new_password'] : '';
        $confirmPassword = isset($_POST['confirm_password']) ? $_POST['confirm_password'] : '';
        
        // Validasi input
        if (empty($currentPassword) || empty($newPassword) || empty($confirmPassword)) {
            setFlashMessage('error', 'Semua field harus diisi');
            return;
        }
        
        if ($newPassword !== $confirmPassword) {
            setFlashMessage('error', 'Password baru dan konfirmasi password tidak cocok');
            return;
        }
        
        // Verifikasi password saat ini
        $userId = $_SESSION['user_id'];
        $user = getUserById($userId);
        
        if (!$user) {
            setFlashMessage('error', 'User tidak ditemukan');
            return;
        }
        
        // Verifikasi password saat ini
        if (!verifyUser($user['username'], $currentPassword)) {
            setFlashMessage('error', 'Password saat ini salah');
            return;
        }
        
        // Ubah password
        if (changePassword($userId, $newPassword)) {
            setFlashMessage('success', 'Password berhasil diubah');
            
            // Redirect ke dashboard
            header('Location: ' . BASE_URL . '/view/dashboard/index.php');
            exit;
        } else {
            setFlashMessage('error', 'Gagal mengubah password');
        }
    }
}

/**
 * Menangani proses reset password
 * Fungsi ini hanya untuk admin
 * 
 * @param int $userId ID user yang akan direset passwordnya
 * @param string $newPassword Password baru
 * @return bool True jika berhasil, false jika gagal
 */
function handleResetPassword($userId, $newPassword) {
    // Cek apakah user adalah admin
    if (!isLoggedIn() || $_SESSION['username'] !== 'admin') {
        return false;
    }
    
    // Validasi input
    if (empty($userId) || empty($newPassword)) {
        return false;
    }
    
    // Reset password
    return changePassword($userId, $newPassword);
}

/**
 * Menambahkan user baru
 * Fungsi ini hanya untuk admin
 * 
 * @param string $username Username baru
 * @param string $password Password
 * @return int|false ID user baru jika berhasil, false jika gagal
 */
function handleAddUser($username, $password) {
    // Cek apakah user adalah admin
    if (!isLoggedIn() || $_SESSION['username'] !== 'admin') {
        return false;
    }
    
    // Validasi input
    if (empty($username) || empty($password)) {
        return false;
    }
    
    // Hash password
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
    
    global $conn;
    
    // Sanitasi input
    $username = sanitizeInput($conn, $username);
    
    // Cek apakah username sudah ada
    $sql = "SELECT id FROM users WHERE username = '$username'";
    $result = executeQuery($conn, $sql);
    
    if (mysqli_num_rows($result) > 0) {
        return false; // Username sudah ada
    }
    
    // Tambahkan user baru
    $sql = "INSERT INTO users (username, password) VALUES ('$username', '$hashedPassword')";
    executeQuery($conn, $sql);
    
    // Return ID user baru
    return mysqli_insert_id($conn);
}

/**
 * Mendapatkan semua user
 * Fungsi ini hanya untuk admin
 * 
 * @return array|false Data semua user jika user adalah admin, false jika bukan
 */
function getAllUsers() {
    // Cek apakah user adalah admin
    if (!isLoggedIn() || $_SESSION['username'] !== 'admin') {
        return false;
    }
    
    global $conn;
    
    $sql = "SELECT id, username, created_at FROM users ORDER BY username ASC";
    $result = executeQuery($conn, $sql);
    
    return fetchData($result);
}

/**
 * Menghapus user
 * Fungsi ini hanya untuk admin dan tidak dapat menghapus diri sendiri
 * 
 * @param int $userId ID user yang akan dihapus
 * @return bool True jika berhasil, false jika gagal
 */
function handleDeleteUser($userId) {
    // Cek apakah user adalah admin
    if (!isLoggedIn() || $_SESSION['username'] !== 'admin') {
        return false;
    }
    
    // Cek apakah user mencoba menghapus dirinya sendiri
    if ($_SESSION['user_id'] == $userId) {
        return false;
    }
    
    global $conn;
    
    $sql = "DELETE FROM users WHERE id = $userId";
    executeQuery($conn, $sql);
    
    return mysqli_affected_rows($conn) > 0;
}